package main

import (
	"fmt"
	"net"

	"github.com/google/go-tpm/tpmutil"
)

type dialer func(network, path string) (net.Conn, error)

type TcpReadWriteCloser struct {
	addr   string
	conn   net.Conn
	dialer dialer
}

func NewTcpReadWriteCloser(addr string) *TcpReadWriteCloser {
	return &TcpReadWriteCloser{
		addr:   addr,
		dialer: net.Dial,
	}
}

func (rwc *TcpReadWriteCloser) Send(input []byte) ([]byte, error) {
	return tpmutil.RunCommandRaw(rwc, input)
}

func (rwc *TcpReadWriteCloser) Read(p []byte) (int, error) {
	if rwc.conn == nil {
		return 0, fmt.Errorf("must call Write then Read in an alternating sequence")
	}
	n, err := rwc.conn.Read(p)
	rwc.conn.Close()
	rwc.conn = nil
	return n, err
}

func (rwc *TcpReadWriteCloser) Write(p []byte) (int, error) {
	if rwc.conn != nil {
		return 0, fmt.Errorf("must call Write then Read in an alternating sequence")
	}
	var err error
	rwc.conn, err = rwc.dialer("tcp", rwc.addr)
	if err != nil {
		return 0, err
	}
	return rwc.conn.Write(p)
}

func (rwc *TcpReadWriteCloser) Close() error {
	if rwc.conn == nil {
		return fmt.Errorf("cannot call Close when no connection is open")
	}
	err := rwc.conn.Close()
	rwc.conn = nil
	return err
}
